/* 

  This is an example of cartoon rendering using the CanvasGL
  construct.  I am indebted to Sami Hamlaoui for both the
  NeHe tutorial regarding this and for the mesh (although it's
  not his).

  All that cartoon shading does is replace normal diffuse lighting
  with discretized texture lookups.  If the texture were correctly
  setup, it would perform similar to normal diffuse lighting.

  My implementation uses OpenGL state to find out how lighting
  should be done on the model.  For multiple lights, it will
  write to the frame buffer with the first light, and then add to
  the color for the subsequent lights.  Then it multiplies the
  texture onto the framebuffer to get the result.  It doesn't exploit
  multitexturing, but that's not a difficult task.

  It also does anti-aliased shadowing.  The shadow itself is just a
  flattened model; then a second pass is run with line mode enabled.
  Using edge-sharing information in the same way as stencil shadowing
  would eliminate interior edges.  I use backface culling, which thus
  requires a manifold mesh (no holes unattached to anything).

  For the lines themselves I just redraw the model with
  line mode on and inverted face culling.  This makes edge changes a
  little sharp, but tends to work very effectively and requires no
  processing, although again using edge-sharing information would
  eliminate interior edges.

  The shadows simply overlap each other, but you can get aliased
  plane shadows that are partially transparent by using the stencil
  buffer.  Get it to test for equality and replace on write and set
  the reference value to nonzero, so that only one draw on this pixel
  wins.  Anti-aliasing the edges is also possible, but it tends to 
  result in bad polygon edges.  I halve the opacity when transparent
  shadows are enabled to reduce the problem.

 */

import net.BurtonRadons.dig.main;
import net.BurtonRadons.dig.gl;

import file;

const char [] register = "digCartoonExample.";

GLuint aUInt (inout ubyte *d)
{
    GLuint v = d [0] | (d [1] << 8) | (d [2] << 16) | (d [3] << 24);

    d += 4;
    return v;
}

GLfloat aFloat (inout ubyte *d)
{
    GLuint v = aUInt (d);

    printf (""); /* Necessary to spoof an optimiser bug. */
    return *(GLfloat *) &v;
}

class CartoonDemoFrame : CanvasGL
{
    GLtexture texture;

    int transparentShadows = false; /* Make transparent shadows. */
    int rotation = true; /* Do rotation. */
    int texturing = true; /* Do texturing. */
    int shadowing = true; /* Do shadowing. */
    int outlineSmooth = true; /* Whether to use smooth lines. */
    int outlineDraw = true; /* Whether to draw the outline. */
    GLfloat outlineWidth = 2; /* Width of the outline lines. */
    Color outlineColor = { 0, 0, 0, 255 }; /* Color of the outlines. */

    int light0 = true;
    int light1 = true;
    int light2 = true;

    Color light0Diffuse = { 84, 57, 46 };
    Color light1Diffuse = { 108, 101, 49 };
    Color light2Diffuse = { 143, 209, 120 };

    GLtexture modelShader; /* Cel shader for the model. */
    GLfloat modelAngle = 0; /* Model angle in degrees. */

    GLfloat speedMax = 90; /* Maximum degrees per second. */
    GLfloat speed = 90; /* Rotation speed (degrees per second). */
    ulong last; /* Last calculated frame. */

    struct Vertex
    {
        vec3 n; /* normal */
        vec2 t; /* texel */
        vec3 v; /* vertex */
    }

    struct Polygon
    {
        Vertex v [3]; /* vertices */
    }

    Polygon [] polygons; /* List of polygons. */

    this (Control parent)
    {
        super (parent);
        left (10);
        top (10);
        suggestWidthAndHeight (512, 512);
        onPaint.add (&paintFrame);

        modelRead ((ubyte []) read ("net/BurtonRadons/dig/examples/cartoon.mdl"));
        createShader ("01225555555555888888888888889999");

        with (texture = gl.textureNew ())
        {
            BrushColor cbrush = new BrushColor (SColor1 (1, 1, 1));//0.8, 0.4));
            BrushNoise nbrush = new BrushNoise (cbrush);
            int w = 128;
            int h = 128;
            GLubyte [] data;

            nbrush.scale (64);
            nbrush.uniform (true);

            data = new GLubyte [w * h * 3];

            for (int x; x < w; x ++)
            for (int y; y < h; y ++)
            {
                Color c = nbrush.eval (x, y);

                data [(x + y * w) * 3 + 0] = c.r;
                data [(x + y * w) * 3 + 1] = c.g;
                data [(x + y * w) * 3 + 2] = c.b;
            }

            image (w, h, GL.RGB, GL.UNSIGNED_BYTE, data);
            minFilter (GL.LINEAR);
        }
        
        gl.lightPosition (GL.LIGHT0, 1, 0, 1);
        gl.lightPosition (GL.LIGHT1, -2, 0, 1);
        gl.lightPosition (GL.LIGHT2, 0, 0, -3);

        with (registry)
        {
            loadColor (register ~ "light0_diffuse", light0Diffuse);
            loadColor (register ~ "light1_diffuse", light1Diffuse);
            loadColor (register ~ "light2_diffuse", light2Diffuse);
            loadInt (register ~ "light0", light0);
            loadInt (register ~ "light1", light1);
            loadInt (register ~ "light2", light2);
            loadInt (register ~ "transparentShadows", transparentShadows);
            loadInt (register ~ "rotation", rotation);
            loadInt (register ~ "texturing", texturing);
            loadInt (register ~ "shadowing", shadowing);
            loadInt (register ~ "outlineSmooth", outlineSmooth);
            loadInt (register ~ "outlineDraw", outlineDraw);
            loadFloat (register ~ "outlineWidth", outlineWidth);
            loadColor (register ~ "outlineColor", outlineColor);
            loadFloat (register ~ "modelAngle", modelAngle);
            loadFloat (register ~ "speedMax", speedMax);
            loadFloat (register ~ "speed", speed);
        }
    }

    /* Read in the model from the data.
     */

    void modelRead (ubyte [] data)
    {
        ubyte *d = data;
        uint count;

        count = aUInt (d);
        polygons.length = count;
        for (int c; c < count; c ++)
        {
            Polygon *p = &polygons [c];

            for (int e; e < 3; e ++)
            {
                Vertex *r = &p.v [e];
                vec3 *v = &r.v;
                vec3 *n = &r.n;
                vec2 *t = &r.t;

                n.x = aFloat (d);
                n.y = aFloat (d);
                n.z = aFloat (d);

                v.x = aFloat (d);
                v.y = aFloat (d);
                v.z = aFloat (d);

                t.x = (rand () % 1024) / 1024.0;
                t.y = (rand () % 1024) / 1024.0;
            }

            p.v [1].t.x = p.v [0].t.x + p.v [1].v.distance (p.v [0].v) * 4;
            p.v [1].t.y = p.v [0].t.y;

            p.v [2].t.x = p.v [0].t.x;
            p.v [2].t.y = p.v [0].t.y + p.v [2].v.distance (p.v [0].v) * 4;
        }
    }

    /* Create and set the cartoon shader texture, a 1D luminance
     * texture showing responsiveness to light angle.  Each character
     * is from 0 to 9, which is the intensity at that point.  The
     * first point is edge on to the light, the last point is face on.
     */
    
    void createShader (char [] content)
    {
        GLubyte [] data; /* The actual content. */
        GLtexture texture; /* The texture object. */
        
        /* Allocate the data. */
        data.length = content.length;
        
        /* Load up the data. */
        for (int c; c < content.length; c ++)
        {
            assert (content [c] >= '0' && content [c] <= '9');
            data [c] = (content [c] - '0') * 255 / 9;
        }
        
        /* Create the texture. */
        with (texture = gl.textureNew ())
        {
            image (content.length, 1, GL.LUMINANCE, GL.UNSIGNED_BYTE, data);
            magFilter (GL.NEAREST); /* Set linear filtering. */
            minFilter (GL.LINEAR); /* Set linear filtering. */
            wrapS (GL.CLAMP);
        }
        
        /* Free the allocated data. */
        delete data; 
        
        /* Store and finish. */
        if (modelShader)
            delete modelShader;
        modelShader = texture;
    }

    void drawShadow (vec3 light, Color shadow)
    {
        Polygon *pStart = polygons;
        Polygon *pEnd = pStart + polygons.length;
        mat4 modelview = gl.modelviewMatrix ();

        modelview = modelview.transpose ();
        light = modelview.rotate (light);

        gl.color (shadow);
        light = light.normalize ();

        Color background = (cast (Frame) parent ()).backgroundColor ();

        /* Enabling culling means that we require a manifold mesh. */
        gl.enable (GL.CULL_FACE);
        gl.cullFace (GL.BACK);

        gl.begin (GL.TRIANGLES);
        for (Polygon *p = pStart; p < pEnd; p ++)
        {
            for (int j; j < 3; j ++)
            {
                vec3 v = p.v [j].v;
                
                gl.vertex (v.x - (v.y + 0.5) * light.x, -0.5, v.z - (v.y + 0.5) * light.z);
            }
        }
        gl.end ();
    }

    void paintFrame (Event e)
    {
        Polygon *pStart = polygons;
        Polygon *pEnd = pStart + polygons.length;
        mat4 modelview; /* Modelview matrix. */
        Color background = (cast (Frame) parent ()).backgroundColor ();

        ulong time = elapsedTime ();
        ulong delta = time - last;

        if (last == 0)
            delta = 0;
        last = time;

        if (rotation)
            speed = speed + delta / 1000.0 * speedMax / 0.5;
        else
            speed = speed - delta / 1000.0 * speedMax / 0.5;

        speed = speed < 0 ? 0 : speed > speedMax ? speedMax : speed;
        modelAngle = modelAngle + speed * delta / 1000.0;

        if (!visible ())
            goto finish;

    /* Setup for painting. */
        beginPaint (); /* Start painting. */

        with (gl)
        {
            gl.store (GL.LIGHT0, light0);
            gl.store (GL.LIGHT1, light1);
            gl.store (GL.LIGHT2, light2);

            gl.lightDiffuse (GL.LIGHT0, light0Diffuse);
            gl.lightDiffuse (GL.LIGHT1, light1Diffuse);
            gl.lightDiffuse (GL.LIGHT2, light2Diffuse);

            gl.clearColor (background); /* Set the clear color to the window background. */
            gl.clear (GL.COLOR_BUFFER_BIT | GL.DEPTH_BUFFER_BIT); /* Clear the depth and color buffers. */

        /* Clear to a handsome background. */
            loadIdentity (PROJECTION);
            loadIdentity (MODELVIEW);
            disable (DEPTH_TEST);
            disable (CULL_FACE);
            disable (BLEND);

            gl.begin (QUADS);
            color (background.blend (Color.Black, 128)); vertex (-1, 1);
            color (background.blend (Color.Black, 160)); vertex (1, 1);
            color (background); vertex (1, 0);
            color (background); vertex (-1, 0);
            end ();
        
        /* Setup GL parameters. */
            enable (DEPTH_TEST); /* Turn on the depth buffer. */
            depthFunc (LEQUAL); /* Store the depth comparison function. */
            store (LINE_SMOOTH, outlineSmooth); /* Set the line smoothing value. */
            disable (LIGHTING); /* Disable lighting. */

            enable (CULL_FACE); /* Enable face culling. */
            cullFace (BACK); /* Cull the back-winding face. */

        /* Setup the matrices. */
            matrixMode (PROJECTION);
            loadIdentity ();
	        perspective (45.0f, width () / (GLfloat) height (), 1.0, 100.0);

            matrixMode (MODELVIEW); /* Set the matrix to MODELVIEW. */
            loadIdentity (); /* Clear to identity. */

            translate (0, 0, -2); /* Push the object into the frame. */
            rotate (modelAngle / 2, 0, 1, 0); /* Rotate the model. */
            modelview = modelviewMatrix (); /* Get the modelview matrix. */

        /* Draw the shadow. */
            if (shadowing)
            {
                int lightCount = 0;

                disable (DEPTH_TEST);

                for (int c = maxLights () - 1; c >= 0; c --)
                {
                    GLenum index = LIGHT0 + c;

                    if (!isEnabled (index))
                        continue;

                    if (transparentShadows && lightCount)
                    {
                        if (lightCount == 1)
                            clear (STENCIL_BUFFER_BIT);
                        stencilFunc (NOTEQUAL, lightCount + 1);
                        stencilOp (KEEP, KEEP, REPLACE);
                        enable (STENCIL_TEST);
                    }

                    vec3 position = gl.lightPosition (index);
                    vec3 angle = position.normalize ();
                    Color diffuse = gl.lightDiffuse (index);

                    Color shadow = AColor (255 - diffuse.intensity (), 255 - diffuse.intensity (), 255 - diffuse.intensity ());
                
                    if (transparentShadows && lightCount)
                        shadow = AColor (shadow.r, shadow.g, shadow.b, 128);
                    else
                        shadow = shadow.blend (background, 128);

                    enable (BLEND);
                    blendFunc (SRC_ALPHA, ONE_MINUS_SRC_ALPHA);

                    drawShadow (angle, shadow);

                    if (outlineSmooth)
                    {
                        if (transparentShadows && lightCount)
                            shadow = AColor (shadow.r, shadow.g, shadow.b, shadow.a / 2);
                        lineWidth (1);
                        polygonMode (FRONT, LINE);
                        drawShadow (angle, shadow);
                        polygonMode (FRONT, FILL);
                    }

                    lightCount ++;
                }

                /* Restore state. */
                disable (STENCIL_TEST);
                enable (DEPTH_TEST);
            }

        /* Find the lights and draw them. */
            int lightCount = 0;

            for (int c = maxLights () - 1; c >= 0; c --)
            {
                GLenum index = LIGHT0 + c;

                if (!isEnabled (index))
                    continue;

                if (lightCount)
                {
                    enable (BLEND);
                    blendFunc (ONE, ONE);
                }
                else
                    disable (BLEND);

                vec3 position = gl.lightPosition (index);
                vec3 angle = position.normalize ();

                lightCount ++;

                modelShader.enable (); /* Enable the shader texture. */
                color (gl.lightDiffuse (index)); /* Set the color of the light. */

                gl.begin (TRIANGLES);
                for (int i; i < polygons.length; i ++)
                {
                    Polygon p = polygons [i];

                    for (int j; j < 3; j ++)
                    {
                        vec3 n = modelview.rotate (p.v [j].n);
                        GLfloat shade;

                        n = n.normalize ();
                        shade = n.dot (angle);
                        if (shade < 0)
                            shade = 0;
                        texel (shade, 0);
                        vertex (p.v [j].v);
                    }
                }
                end ();
            }

        /* Draw the model. */
            if (texturing)
            {
                if (lightCount)
                {
                    enable (BLEND);
                    blendFunc (DST_COLOR, ZERO);
                }
                else
                    disable (BLEND);

                texture.enable ();
                color (Color.White);

                gl.begin (TRIANGLES);
                for (Polygon *p = pStart; p < pEnd; p ++)
                {
                    for (int j; j < 3; j ++)
                    {
                        vec3 v = p.v [j].v;
                        vec3 n = p.v [j].n;
                        vec2 t = p.v [j].t;

                        texel (t);//(v.x - v.z) * 4, (v.y + v.z) * 4);
                        vertex (v);
                    }
                }
                end ();
            }


        /* Draw the outlines. */
            disable (TEXTURE_2D); /* No texturing now. */

            if (outlineDraw)
            {
                enable (BLEND); /* Enable blending. */
                blendFunc (SRC_ALPHA, ONE_MINUS_SRC_ALPHA); /* Set the blending mode. */
                polygonMode (BACK, LINE); /* Draw backfacing polygons as wire. */
                lineWidth (outlineWidth); /* Set the outline width. */
                cullFace (FRONT); /* Cull the front-winding face. */
                color (outlineColor);

                gl.begin (TRIANGLES);
                for (int i; i < polygons.length; i ++)
                {
                    for (int j; j < 3; j ++)
                        vertex (polygons [i].v [j].v);
                }
                end ();

                polygonMode (BACK, FILL); /* Restore the polygon mode. */
            }
        }

        /* Finished. */
            endPaint (); /* Finish painting, swap buffers. */

    finish:
        timer (10, &paintFrame);
    }
}

/* This sample shows how to duplicate control behaviour semi-portably. */
class FakeButton : Canvas
{
    Dispatcher onClick; /**< Dispatched when the button is clicked upon. */
    Dispatcher onDClick; /**< Dispatched when the button is double-clicked upon; if this is empty, #onClick is used instead. */

    this (Control parent)
    {
        super (parent);
        onPaint.add (&paint);
        onLButtonDown.add (&lbuttonDown);
        onLButtonUp.add (&lbuttonUp);
        onLostFocus.add (&lostFocus);
        onMouseMove.add (&mouseMove);
        doubleBuffer (false); // Perhaps a little TOO accurate. :-)
        sfont = super.font ();
    }

    void paint (Event e)
    {
        int sx = 0, sy = 0, ex = width (), ey = height ();
        int tx, ty;

        beginPaint ();
        font (sfont);
        buttonDraw (sx, sy, ex, ey, pressed, isFocus ());
        buttonClientRegion (sx, sy, ex, ey);
        tx = sx, ty = sy;

        if (sAlign == '<')
            textAlign ("<");
        else if (sAlign == '>')
        {
            tx = ex;
            textAlign (">");
        }
        else
        {
            textAlign ("|");
            tx = (ex - sx) / 2 + sx;
        }

        if (pressed)
            buttonPressedOffset (tx, ty);

        super.textPrint (tx, ty, text);

        endPaint ();
    }

    void lbuttonDown (Event e)
    {
        makeFocus ();
        pressed = true;
        super.paint ();
        captureMouse ();
    }

    void lbuttonUp (Event e)
    {
        if (!isCaptor ())
            return;
        setPressed (false);
        releaseMouse ();
        if (inClientRegion (e.x, e.y))
            onClick.notify (e);
    }

    void lostFocus (Event e)
    {
        super.paint ();
    }

    void mouseMove (Event e)
    {
        if (!isCaptor ())
            return;
        setPressed (inClientRegion (e.x, e.y));
    }

    void font (Font f)
    {
        sfont = f;
        recalculate ();
    }

    void caption (char [] text)
    {
        this.text = text;
        recalculate ();
    }

    /* Set text alignment - "<" for left, ">" for right, "|" for center. */
    void alignment (char value)
    {
        sAlign = value;
    }

protected:
    Font sfont = null;
    char sAlign = '|'; /* Alignment value. */
    char [] text; /* Caption text. */
    bit pressed = false; /* Set if currently down. */

    /* Set pressed and repaint if changed. */
    void setPressed (bit value)
    {
        if (pressed == value)
            return;
        pressed = value;
        super.paint ();
    }

    /* Recalculate size. */
    void recalculate ()
    {
        Font f = sfont;
        int width, height;
        int bwidth, bheight;

        f.textExtent (this, text, width, height);
        buttonBorder (bwidth, bheight);
        suggestWidthAndHeight (width + bwidth, height + bheight);
        this.text = text;
    }
}

class CartoonDemo : Frame
{
    CartoonDemoFrame frame; /* The GL frame to render into. */

    ColorSelector colorSelector;
    CheckBox texturing;
    CheckBox shadowing;
    CheckBox transparentShadows;
    CheckBox edging;
    CheckBox smoothlines;
    CheckBox rotation;
    CheckBox light0;
    CheckBox light1;
    CheckBox light2;

    Canvas lineWidth;
    Canvas light0Color;
    Canvas light1Color;
    Canvas light2Color;

    ComboBox celShader;
    Spinner outlineWidth;

    Bitmap bitmap;

    this ()
    {
        int r = 0;

        caption ("Cartoon demo");
        maximizable (true);
        minimizable (true);
        resizable (true);

        with (frame = new CartoonDemoFrame (this))
        {
            grid (0, 0, 1, 10);
            sticky ("<>^v");
        }

        colorSelector = new ColorSelector ();

        with ((rotation = new CheckBox (this)))
        {
            caption ("Model Rotation");
            onClick.add (&toggleRotation);
            checked ((bit) frame.rotation);
            gridAddRow (1, r, 2, 1);
        }

        with ((texturing = new CheckBox (this)))
        {
            caption ("Texturing");
            onClick.add (&toggleTexturing);
            checked ((bit) frame.texturing);
            gridAddRow (1, r, 2, 1);
        }

        with ((shadowing = new CheckBox (this)))
        {
            caption ("Shadowing");
            onClick.add (&toggleShadowing);
            checked ((bit) frame.shadowing);
            gridAddRow (1, r, 2, 1);
        }

        with ((transparentShadows = new CheckBox (this)))
        {
            caption ("Transparent Shadows");
            onClick.add (&toggleTransparentShadows);
            checked ((bit) frame.transparentShadows);
            gridAddRow (1, r, 2, 1);
        }

        with ((edging = new CheckBox (this)))
        {
            caption ("Outline Edging");
            onClick.add (&toggleEdging);
            checked ((bit) frame.outlineDraw);
            gridAddRow (1, r, 2, 1);
        }

        with ((smoothlines = new CheckBox (this)))
        {
            caption ("Smooth lines");
            onClick.add (&toggleOutlineSmooth);
            checked ((bit) frame.outlineSmooth);
            gridAddRow (1, r, 2, 1);
        }

        with ((light0 = new CheckBox (this)))
        {
            caption ("First light");
            onClick.add (&toggleLight0);
            checked ((bit) frame.light0);
            gridAddRow (2, r);
        }

        with ((light1 = new CheckBox (this)))
        {
            caption ("Second light");
            onClick.add (&toggleLight1);
            checked ((bit) frame.light1);
            gridAddRow (2, r);
        }

        with ((light2 = new CheckBox (this)))
        {
            caption ("Third light");
            onClick.add (&toggleLight2);
            checked ((bit) frame.light2);
            gridAddRow (2, r);
        }

        with ((light0Color = new Canvas (this)))
        {
            width (light0.height ());
            height (light0.height ());
            onPaint.add (&paintLight0Color);
            onLButtonUp.add (&setLight0Color);
            grid (1, light0.gridRow ());
        }

        with ((light1Color = new Canvas (this)))
        {
            width (light1.height ());
            height (light1.height ());
            onPaint.add (&paintLight1Color);
            onLButtonUp.add (&setLight1Color);
            grid (1, light1.gridRow ());
        }

        with ((light2Color = new Canvas (this)))
        {
            width (light2.height ());
            height (light2.height ());
            onPaint.add (&paintLight2Color);
            onLButtonUp.add (&setLight2Color);
            grid (1, light2.gridRow ());
        }

        with ((celShader = new ComboBox (this)))
        {
            gridAddRow (1, r, 2, 1);
            //sticky ("<>");
            opAdd ("cartoon", "Cartoon Shader", &cartoonShader);
            current ("cartoon");
            opAdd ("diffuse", "Diffuse Shader", &diffuseShader);
            opAdd ("rainbow", "Rainbow Shader", &rainbowShader);
            opAdd ("flat", "Flat Shader", &flatShader);

            char [] cur;

            if (registry.loadString (register ~ "shader", cur))
                setShader (cur);
        }

        with (outlineWidth = new Spinner (this))
        {
            gridAddRow (1, r, 2, 1);
            //sticky ("<>");
            caption ("Line Width:");
            range (gl.lineWidthMin (), gl.lineWidthMax (), 0.1);
            value (frame.outlineWidth);
            onChange.add (&setOutlineWidth);
        }

        frame.grid (0, 0, 1, r);

        //onResized.add (&resize);
        //onMaximized.add (&resize);
        //display ();
    }

    void setOutlineWidth (float value)
    {
        frame.outlineWidth = value;
        registry.saveFloat (register ~ "outlineWidth", frame.outlineWidth);
    }

    void resize (Event e)
    {
        static bit recurse = false;

        if (recurse)
            return;
        recurse = true;

        int cx = e.x - width ();
        int cy = e.y - height ();

        frame.width (frame.width () + cx);
        frame.height (frame.height () + cy);

        //paint ();

        recurse = false;
        display ();
    }

    void setShader (char [] name)
    {
        if (!celShader.hasItem (name))
            return;
        celShader.currentAndNotify (name);
    }

    void cartoonShader (Event e)
    {
        frame.makeCurrent ();
        frame.createShader ("01225555555555888888888888889999");
        registry.saveString (register ~ "shader", "cartoon");
    }

    void flatShader (Event e)
    {
        frame.makeCurrent ();
        frame.createShader ("0099999999999999");
        registry.saveString (register ~ "shader", "flat");
    }

    void diffuseShader (Event e)
    {
        registry.saveString (register ~ "shader", "diffuse");
        with (frame)
        {
            const int w = 16;
            GLubyte [w * 3] data;

            makeCurrent ();
            if (modelShader)
                delete modelShader;

            for (int c; c < w; c ++)
            {
                int i = c * 255 / w;

                data [c * 3 + 0] = data [c * 3 + 1] = data [c * 3 + 2] = i;
            }

            with (modelShader = gl.textureNew ())
            {
                image (w, 1, GL.RGB, GL.UNSIGNED_BYTE, data);
                magFilter (GL.LINEAR);
                wrapS (GL.CLAMP);
            }
        }
    }

    void rainbowShader (Event e)
    {
        registry.saveString (register ~ "shader", "rainbow");
        with (frame)
        {
            Color [16] colors;

            makeCurrent ();
            if (modelShader)
                delete modelShader;

            for (int c; c < colors.length; c ++)
            {
                float h = c / (float) colors.length;
                float s = 0.7;
                float v = 1.0;

                colors [c] = Color.fromHSV (h, s, v);
            }

            with (modelShader = gl.textureNew ())
                image (colors.length, 1, GL.RGBA, GL.UNSIGNED_BYTE, colors);
        }
    }

    void toggleTexturing (Event e)
    {
        texturing.checked ((bit) (frame.texturing = !frame.texturing));
        registry.saveInt (register ~ "texturing", frame.texturing);
    }

    void toggleShadowing (Event e)
    {
        shadowing.checked ((bit) (frame.shadowing = !frame.shadowing));
        registry.saveInt (register ~ "shadowing", frame.shadowing);
    }

    void toggleEdging (Event e)
    {
        edging.checked ((bit) (frame.outlineDraw = !frame.outlineDraw));
        registry.saveInt (register ~ "outlineDraw", frame.outlineDraw);
    }

    void toggleRotation (Event e)
    {
        rotation.checked ((bit) (frame.rotation = !frame.rotation));
        registry.saveInt (register ~ "rotation", frame.rotation);
    }

    void toggleOutlineSmooth (Event e)
    {
        smoothlines.checked ((bit) (frame.outlineSmooth = !frame.outlineSmooth));
        registry.saveInt (register ~ "outlineSmooth", frame.outlineSmooth);
    }

    void toggleLight0 (Event e)
    {
        light0.checked ((bit) (frame.light0 = !frame.light0));
        registry.saveInt (register ~ "light0", frame.light0);
    }

    void toggleLight1 (Event e)
    {
        light1.checked ((bit) (frame.light1 = !frame.light1));
        registry.saveInt (register ~ "light1", frame.light1);
    }

    void toggleLight2 (Event e)
    {
        light2.checked ((bit) (frame.light2 = !frame.light2));
        registry.saveInt (register ~ "light2", frame.light2);
    }

    void toggleTransparentShadows (Event e)
    {
        transparentShadows.checked ((bit) (frame.transparentShadows = !frame.transparentShadows));
        registry.saveInt (register ~ "transparentShadows", frame.transparentShadows);
    }

    void paintLight0Color (Event e)
    {
        with (light0Color)
        {
            beginPaint ();
            clear (frame.light0Diffuse);
            endPaint ();
        }
    }

    void paintLight1Color (Event e)
    {
        with (light1Color)
        {
            beginPaint ();
            clear (frame.light1Diffuse);
            endPaint ();
        }
    }

    void paintLight2Color (Event e)
    {
        with (light2Color)
        {
            beginPaint ();
            clear (frame.light2Diffuse);
            endPaint ();
        }
    }

    void setLight0Color (Event e)
    {
        with (colorSelector)
        {
            color = frame.light0Diffuse;
            if (run ())
            {
                frame.light0Diffuse = color;
                light0Color.paint ();
                registry.saveColor (register ~ "light0_diffuse", frame.light0Diffuse);
            }
        }
    }

    void setLight1Color (Event e)
    {
        with (colorSelector)
        {
            color = frame.light1Diffuse;
            if (run ())
            {
                frame.light1Diffuse = color;
                light1Color.paint ();
                registry.saveColor (register ~ "light1_diffuse", frame.light1Diffuse);
            }
        }
    }

    void setLight2Color (Event e)
    {
        with (colorSelector)
        {
            color = frame.light2Diffuse;
            if (run ())
            {
                frame.light2Diffuse = color;
                light2Color.paint ();
                registry.saveColor (register ~ "light2_diffuse", frame.light2Diffuse);
            }
        }
    }
}

int main (char [] [] argv)
{
    (new CartoonDemo ()).showModal ();
    return 0;
}
